//
// (c) 2020 wesolutions GmbH
// All rights reserved.
//

import QtQuick 2.3
import QtQuick.Dialogs  1.1
import QtQuick.Layouts  1.1

import wesual.Ui                1.0

import wesual.Controls 1.0

import wesual.data.Core         1.0
import wesual.data.FoodService  1.0

import wesual.module.ProductManager 1.0

SidebarItem {
    id : editor

    property FoodServiceProduct product

    readonly property string socialShareQml : "
        import QtQuick 2.3

        import wesual.Editor 1.0

        SocialShareButton {
            id : shareButton
            documents : [ product ]
            anchors.left : picker.right
            anchors.leftMargin : 10
        }"

    signal navigateToProduct()

    panel : SidebarItem.RightScrollablePanel
    active : product !== null
    sortIndex : 0
    width : parent ? parent.width : 100
    height : items.height


    // prevent deselection of the current product on click outside of controls
    MouseArea {
        id : stealClickFromEditor
        width : items.width
        height : items.height
        onClicked : {
            editor.focus = true;
        }
    }

    Column {
        id : items
        width : parent.width
        spacing : 9
        height : childrenRect.height

        Item {
            id : pickerItem

            width :  parent.width
            height : picker.height

            ImagePickerButton {
                id : picker
                width : 110
                height : 110
                imageSource : product && product.primaryImage ?
                                             product.primaryImage.thumbnail : ""
                document : product ? product : null
                onSelectedImages : {
                    product.beginRecordChanges();
                    product.primaryImage = images[0];
                    product.endRecordChanges();
                }

                onNavigateToProduct : {
                    editor.navigateToProduct();
                }
            }

            Component.onCompleted :
                Qt.createQmlObject(socialShareQml, pickerItem)
        }

        GridLayout {
            id : grid

            columns : 2
            width : parent.width
            clip : true

            Label {
                text : product ? product.propertyName("price") : ""
                Layout.maximumWidth : grid.width / 3
            }

            Row {
                spacing : 3
                Layout.fillWidth : true
                Text {
                    anchors.verticalCenter : parent.verticalCenter
                    font {
                        family : UI.fontFamily(UI.PrimaryFont)
                        weight : UI.fontWeight(UI.SecondaryFont)
                        pixelSize : 14
                    }
                    text : {
                        if (!editor.product)
                            return "";

                        var locale = Qt.locale(product.language);

                        return UI.currencyString(product.context.country,
                                                 UI.fontFamily(UI.PrimaryFont));
                    }
                }
                PriceEditor {
                    width : 80
                    height : 24
                    document : editor.product
                    showBorder : false
                    font {
                        family : UI.fontFamily(UI.PrimaryFont)
                        weight : UI.fontWeight(UI.SecondaryFont)
                        pixelSize : 14
                    }
                }
            }
            Label {
                text : product ? product.propertyName("tags") : ""
                Layout.maximumWidth : grid.width / 3
            }
            FoodServiceTagLabelsPicker {
                id : tagPicker

                //Layout.fillWidth : true

                tags : filteredTagListing
                product : editor.product

                hoverNeeded : true
            }
            Item {
                Layout.maximumWidth : grid.width / 3
                height : gridView.implicitHeight
                visible : gridView.implicitHeight > 0
            }

            TagButtonFlow {
                id : gridView

                Layout.fillWidth : true

                tagNamePixelSize : 14
                tagIconSize : 25

                tagListing : {
                    if (!editor.product)
                        return [];

                    var allTags = editor.product.tags;
                    var foodserviceTags = [];

                    for (var i = 0; i < allTags.length; ++i) {
                        if (allTags[i].type === "tag:product:foodservice")
                            foodserviceTags.push(allTags[i]);
                    }

                    return foodserviceTags;
                }

                visible : implicitHeight > 0

                onRemoveTag : {
                    editor.product.beginRecordChanges();
                    editor.product.removeTag(tag);
                    editor.product.endRecordChanges();
                }
            }
            Label {
                text : qsTrId("3c6d3b9a1fbe063f")
                Layout.maximumWidth : grid.width / 3
            }
            Item {
                height : allergenButton.height
                Layout.fillWidth : true

                FoodServiceAllergenButton {
                    id : allergenButton

                    currentProduct  : editor.product
                    allergenListing : sortedAllergens.documents ?
                                      sortedAllergens.resultingDocuments : []

                    maximumWidth : parent.width - 100
                }
            }
            Label {
                text : qsTrId("48f5fac2f7ca2304")
                Layout.maximumWidth : grid.width / 3
            }
            Item {
                Layout.fillWidth : true
                height           : nameEditor.height

                UiCondensedLineEdit {
                    id : nameEditor

                    x        : 6
                    text     : editor.product ? editor.product.name : ""
                    bindable : true
                    onUpdateRequested : {
                        product.beginRecordChanges();
                        product.name = newValue;
                        product.endRecordChanges();
                    }
                }
            }
            Label {
                text :
                    qsTrId("632d35bb1ac3e18a")
                Layout.maximumWidth : grid.width / 3
            }
            Item {
                Layout.fillWidth : true
                height           : descriptionEditor.height

                UiCondensedTextEdit {
                    id : descriptionEditor

                    x        : 6
                    text     : editor.product ? editor.product.description : ""
                    bindable : true
                    onUpdateRequested : {
                        product.beginRecordChanges();
                        product.description = newValue;
                        product.endRecordChanges();
                    }
                }
            }
        }

        Rectangle {
            height : 1
            width  : parent.width
            color  : UI.color(UI.SecondarySeparationLine)
        }

        Component {
            id : editorRow

            Row {
                id : row

                spacing : 8

                UiLabel {
                    width : grid.width / 3
                    text  : modelData.label
                    anchors.verticalCenter : parent.verticalCenter
                }
                Item {
                    width  : childrenRect.width
                    height : childrenRect.height

                    Component.onCompleted : {
                        var comp = modelData.getEditorComponent(
                                    SchemaVariable.CondensedValueEditor);
                        if (comp) {
                            comp.createObject(row, {
                                "variable"   : modelData,
                                "valueStore" : Qt.binding(function() {
                                    return product;
                                })
                            });
                        }
                    }
                }
            }
        }

        Repeater {
            visible : false
            model : product && product.productClass ?
                        product.productClass.variables : []

            delegate : editorRow
        }
    }
    TypeSorter {
        id : nameSorter

        property : "name"
    }
    TypeSorter {
        id : codeSorter

        property : "letterCode"
    }

    TypeListing {
        id : allergens
        context : _we_globalContext || null
        type : "tag:allergen"
    }
    DocumentListSorter {
        id : sortedAllergens

        documents : allergens.documents
        sorters : codeSorter
    }

    TypeListing {
        id : tagListing
        context : _we_globalContext || null
        type : "tag:product:foodservice"
    }
    DocumentListSorter {
        id : sortedTags

        documents : tagListing.documents
        sorters : nameSorter
    }
    DocumentListFilter {
        id : filteredTagListing

        documents : sortedTags.resultingDocuments
    }
}
